<?php
/*
 * Copyright (c) 2025 AltumCode (https://altumcode.com/)
 *
 * This software is licensed exclusively by AltumCode and is sold only via https://altumcode.com/.
 * Unauthorized distribution, modification, or use of this software without a valid license is not permitted and may be subject to applicable legal actions.
 *
 * 🌍 View all other existing AltumCode projects via https://altumcode.com/
 * 📧 Get in touch for support or general queries via https://altumcode.com/contact
 * 📤 Download the latest version via https://altumcode.com/downloads
 *
 * 🐦 X/Twitter: https://x.com/AltumCode
 * 📘 Facebook: https://facebook.com/altumcode
 * 📸 Instagram: https://instagram.com/altumcode
 */

namespace Altum\controllers;

use Altum\Alerts;
use Altum\Models\Website;

defined('ALTUMCODE') || die();

class Pwas extends Controller {

    public function index() {

        \Altum\Authentication::guard();

        /* Prepare the filtering system */
        $filters = (new \Altum\Filters(['user_id',], ['name'], ['pwa_id', 'name', 'last_datetime', 'datetime']));
        $filters->set_default_order_by($this->user->preferences->pwas_default_order_by, $this->user->preferences->default_order_type ?? settings()->main->default_order_type);
        $filters->set_default_results_per_page($this->user->preferences->default_results_per_page ?? settings()->main->default_results_per_page);

        /* Prepare the paginator */
        $total_rows = database()->query("SELECT COUNT(*) AS `total` FROM `pwas` WHERE `user_id` = {$this->user->user_id} {$filters->get_sql_where()}")->fetch_object()->total ?? 0;
        $paginator = (new \Altum\Paginator($total_rows, $filters->get_results_per_page(), $_GET['page'] ?? 1, url('pwas?' . $filters->get_get() . '&page=%d')));

        /* Get the pwas list for the user */
        $pwas = [];
        $pwas_result = database()->query("SELECT * FROM `pwas` WHERE `user_id` = {$this->user->user_id} {$filters->get_sql_where()} {$filters->get_sql_order_by()} {$paginator->get_sql_limit()}");
        while($row = $pwas_result->fetch_object()) {
            $row->settings = json_decode($row->settings ?? '');
            $pwas[] = $row;
        }

        /* Export handler */
        process_export_csv($pwas, 'include', ['pwa_id', 'user_id', 'name', 'name', 'last_datetime', 'datetime'], sprintf(l('pwas.title')));
        process_export_json($pwas, 'include', ['pwa_id', 'user_id', 'name', 'name', 'settings', 'manifest', 'last_datetime', 'datetime'], sprintf(l('pwas.title')));

        /* Prepare the pagination view */
        $pagination = (new \Altum\View('partials/pagination', (array) $this))->run(['paginator' => $paginator]);

        /* Prepare the view */
        $data = [
            'pwas' => $pwas,
            'total_pwas' => $total_rows,
            'pagination' => $pagination,
            'filters' => $filters,
        ];

        $view = new \Altum\View(\Altum\Plugin::get('pwa')->path . 'views/pwas/index', (array) $this, true);

        $this->add_view_content('content', $view->run($data));

    }

    public function bulk() {

        \Altum\Authentication::guard();

        /* Check for any errors */
        if(empty($_POST)) {
            redirect('pwas');
        }

        if(empty($_POST['selected'])) {
            redirect('pwas');
        }

        if(!isset($_POST['type'])) {
            redirect('pwas');
        }

        //ALTUMCODE:DEMO if(DEMO) Alerts::add_error('This command is blocked on the demo.');

        if(!\Altum\Csrf::check()) {
            Alerts::add_error(l('global.error_message.invalid_csrf_token'));
        }

        if(!Alerts::has_field_errors() && !Alerts::has_errors()) {

            set_time_limit(0);

            switch($_POST['type']) {
                case 'delete':

                    /* Team checks */
                    if(\Altum\Teams::is_delegated() && !\Altum\Teams::has_access('delete.pwas')) {
                        Alerts::add_info(l('global.info_message.team_no_access'));
                        redirect('pwas');
                    }

                    foreach($_POST['selected'] as $pwa_id) {
                        db()->where('pwa_id', $pwa_id)->where('user_id', $this->user->user_id)->delete('pwas');
                    }

                    break;
            }

            /* Set a nice success message */
            Alerts::add_success(l('bulk_delete_modal.success_message'));

        }

        redirect('pwas');
    }

    public function delete() {

        \Altum\Authentication::guard();

        /* Team checks */
        if(\Altum\Teams::is_delegated() && !\Altum\Teams::has_access('delete.pwas')) {
            Alerts::add_info(l('global.info_message.team_no_access'));
            redirect('pwas');
        }

        if(empty($_POST)) {
            redirect('pwas');
        }

        $pwa_id = (int) query_clean($_POST['pwa_id']);

        //ALTUMCODE:DEMO if(DEMO) if($this->user->user_id == 1) Alerts::add_error('Please create an account on the demo to test out this function.');

        if(!\Altum\Csrf::check()) {
            Alerts::add_error(l('global.error_message.invalid_csrf_token'));
        }

        if(!$pwa = db()->where('pwa_id', $pwa_id)->where('user_id', $this->user->user_id)->getOne('pwas', ['pwa_id', 'name'])) {
            redirect('pwas');
        }

        if(!Alerts::has_field_errors() && !Alerts::has_errors()) {

            /* Delete the resource */
            db()->where('pwa_id', $pwa_id)->where('user_id', $this->user->user_id)->delete('pwas');

            /* Set a nice success message */
            Alerts::add_success(sprintf(l('global.success_message.delete1'), '<strong>' . $pwa->name . '</strong>'));

            redirect('pwas');
        }

        redirect('pwas');
    }
}
