<?php
/*
 * Copyright (c) 2025 AltumCode (https://altumcode.com/)
 *
 * This software is licensed exclusively by AltumCode and is sold only via https://altumcode.com/.
 * Unauthorized distribution, modification, or use of this software without a valid license is not permitted and may be subject to applicable legal actions.
 *
 * 🌍 View all other existing AltumCode projects via https://altumcode.com/
 * 📧 Get in touch for support or general queries via https://altumcode.com/contact
 * 📤 Download the latest version via https://altumcode.com/downloads
 *
 * 🐦 X/Twitter: https://x.com/AltumCode
 * 📘 Facebook: https://facebook.com/altumcode
 * 📸 Instagram: https://instagram.com/altumcode
 */

namespace Altum\Controllers;

use Altum\Alerts;

defined('ALTUMCODE') || die();

class AdminPingServerUpdate extends Controller {

    public function index() {

        $ping_server_id = isset($this->params[0]) ? (int) $this->params[0] : null;

        if(!\Altum\Plugin::is_installed('ping-servers') && !\Altum\Plugin::is_active('ping-servers') && $ping_server_id !== 1) {
            Alerts::add_error(sprintf(l('admin_plugins.no_access'), \Altum\Plugin::get('ping-servers')->name ?? 'ping-servers'));
            redirect('admin/ping-servers');
        }

        /* Check if user exists */
        if(!$ping_server = db()->where('ping_server_id', $ping_server_id)->getOne('ping_servers')) {
            redirect('admin/ping-servers');
        }

        if(!empty($_POST)) {
            /* Clean some posted variables */
            $_POST['name'] = input_clean($_POST['name']);
            $_POST['url'] = input_clean($_POST['url']);
            $_POST['country_code'] = array_key_exists($_POST['country_code'], get_countries_array()) ? input_clean($_POST['country_code']) : 'US';
            $_POST['city_name'] = input_clean($_POST['city_name']);
            $_POST['is_enabled'] = $ping_server->ping_server_id == 1 ? 1 : (int) isset($_POST['is_enabled']);

            //ALTUMCODE:DEMO if(DEMO) Alerts::add_error('This command is blocked on the demo.');

            /* Check for any errors */
            $required_fields = ['name', 'city_name'];
            foreach($required_fields as $field) {
                if(!isset($_POST[$field]) || (isset($_POST[$field]) && empty($_POST[$field]) && $_POST[$field] != '0')) {
                    Alerts::add_field_error($field, l('global.error_message.empty_field'));
                }
            }

            if(empty($_POST['url']) && $ping_server->ping_server_id != 1) {
                Alerts::add_field_error('url', l('global.error_message.empty_field'));
            }

            if(!\Altum\Csrf::check()) {
                Alerts::add_error(l('global.error_message.invalid_csrf_token'));
            }

            if(!Alerts::has_field_errors() && !Alerts::has_errors()) {

                /* Update the row of the database */
                db()->where('ping_server_id', $ping_server->ping_server_id)->update('ping_servers', [
                    'name' => $_POST['name'],
                    'url' => $_POST['url'],
                    'country_code' => $_POST['country_code'],
                    'city_name' => $_POST['city_name'],
                    'is_enabled' => $_POST['is_enabled'],
                    'last_datetime' => get_date(),
                ]);

                /* Clear the cache */
                cache()->deleteItem('ping_servers');

                /* Set a nice success message */
                Alerts::add_success(sprintf(l('global.success_message.update1'), '<strong>' . $_POST['name'] . '</strong>'));

                /* Request the data from outside source */
                if($ping_server_id != 1) {
                    try {
                        \Unirest\Request::timeout(5 + 3);
                        $response = \Unirest\Request::post($_POST['url'], [], [
                            'user_agent' => settings()->monitors_heartbeats->user_agent,
                            'ping_method' => settings()->monitors_heartbeats->monitors_ping_method,
                            'type' => 'website',
                            'target' => 'https://example.com/',
                            'port' => 0,
                            'settings' => json_encode([
                                'timeout_seconds' => 5,
                                'request_method' => 'get',
                            ]),
                            'debug' => 1
                        ]);
                    } catch (\Exception $exception) {
                        $exception = true;
                        Alerts::add_error(l('admin_ping_servers.error_message') . '<br />' . $exception->getMessage());
                    }

                    if(!isset($exception)) {

                        /* Make sure the values wer got are the proper ones */
                        if(!isset($response->body->is_ok)) {
                            Alerts::add_error(l('admin_ping_servers.error_message') . '<br />' . $response->raw_body);
                        } else {
                            Alerts::add_success(l('admin_ping_servers.success_message'));
                            Alerts::add_success(sprintf(l('admin_ping_servers.success_message2'), '<strong>' . 'https://example.com/' . '</strong>', '<strong>' . $_POST['url'] . '</strong>', $response->raw_body));
                        }

                    }
                }

                redirect('admin/ping-server-update/' . $ping_server->ping_server_id);
            }

        }

        /* Main View */
        $data = [
            'ping_server' => $ping_server
        ];

        $view = new \Altum\View('admin/ping-server-update/index', (array) $this);

        $this->add_view_content('content', $view->run($data));

    }

}
